# utils.py
"""
Contains helper functions for the simulation.
"""

import numpy as np

def discretize(value, num_bins, value_range):
    """
    Discretizes a continuous value into a bin index.
    """
    low, high = value_range
    if value <= low: return 0
    if value >= high: return num_bins - 1
    bin_size = (high - low) / num_bins
    # Ensure the bin index is within bounds
    return min(int((value - low) / bin_size), num_bins - 1)

def get_report_from_action(action_index, num_actions, report_range):
    """
    Converts a discrete action index back to a report value.
    """
    low, high = report_range
    if num_actions <= 1: return low # Avoid division by zero
    # Calculate the report value based on the action index
    return low + action_index * (high - low) / (num_actions - 1)

def get_action_from_report(report, num_actions, report_range):
    """
    Converts a report value to the closest discrete action index.
    """
    low, high = report_range
    if report <= low: return 0
    if report >= high: return num_actions - 1
    # Avoid division by zero if high equals low
    if high == low: return 0
    # Calculate the floating-point action representation
    action_float = (report - low) / ((high - low) / (num_actions - 1))
    # Round to the nearest integer and clamp within valid range
    return min(max(0, int(round(action_float))), num_actions - 1)

def project_non_negative(v):
    """Projects a vector onto the non-negative orthant."""
    return np.maximum(v, 0)